'use strict';
define(function () {
  var itvsoumFactory = function ($http, $rootScope, $filter,
    gclayers, FeatureTypeFactory, $q, gcRestrictionProvider,
    $timeout, IndigauFactory, gaDomUtils, itvUtilsFactory, gaJsUtils
  ) {

    const portalid = itvUtilsFactory.getPortalId();
    let app = angular.module('gcMain').app;

    const getApp = () => {
      app = localStorage.getItem('app');
      if (!app) {
        $timeout(getApp, 500);
      }
    };

    if (app == undefined || app == '') {
      getApp();
    }


    const user = $rootScope.xgos.user;

    const lyrNames = ['EVT_Inspection', 'EVT_Inspection_Part',
      'EVT_Inspection_Part_Detail', 'EVT_Inspection_Part_Entete', 'CAJ_REL_AR'];

    const itvsmobj = {};


    var EVT_full_Inspection = {
      EVT_Inspection: gclayers.getOperationalLayerByName('EVT_Inspection'),
      EVT_Inspection_Part: gclayers.getOperationalLayerByName(
        'EVT_Inspection_Part'
      ),
      EVT_Inspection_Part_Detail: gclayers.getOperationalLayerByName(
        'EVT_Inspection_Part_Detail'
      ),
      EVT_Inspection_Part_Entete: gclayers.getOperationalLayerByName(
        'EVT_Inspection_Part_Entete'
      ),
    };


    /**
     * Retourne VRAI si le FTI est celui cherché.
     * Pour gérer le cas ArcGIS, on passe par la configuration.
     *
     * @param {*} fti: FTI à tester
     * @param {*} conf: Configuration pour obtenir les composants ArcGIS
     * @param {*} searched: Nom du FTI cherché
     * @returns: VRAI si le FTI est celui cherché, FAUX sinon
     */
    const componentMatchesSearched = (fti, conf, searched) => {
      return fti.name === searched ||
        (conf.arcgiscomponent && conf.arcgiscomponent[searched]
          && fti.name == conf.arcgiscomponent[searched].name);
    };


    const setEVT_full_Inspection = (conf) => {
      var promise = FeatureTypeFactory.get();
      promise.then((res) => {
        // adapte aux modifications de FeatureFactory.get
        // (doit utiliser res et non plus res.data)
        let done = 0;
        for (let fti of res) {
          for (let ind = 0; ind < lyrNames.length; ind++) {
            if (componentMatchesSearched(fti, conf, lyrNames[ind])) {
              EVT_full_Inspection[lyrNames[ind]] = fti;
              done++;
              break;
            }
          }
          if (done == lyrNames.length) {
            break;
          }
        }
      });

      return promise;
    };


    function getEVT_full_Inspection() {
      return EVT_full_Inspection;
    }

    function saveInspectionProperties(senddata, itvid) {
      var promise = $http.post(
        '/services/' +
        portalid +
        '/itv/{appname}/saveInspectionProperties?f=json&itvid=' +
        itvid,
        senddata
      );
      return promise;
    }

    function getInspectionProperties(itvid) {
      var promise = $http.get(
        '/services/' +
        portalid +
        '/itv/{appname}/getInspectionProperties?f=json&itvid=' +
        itvid
      );
      return promise;
    }


    const initDb = (dsname, conf, srid) => {
      //-- Le nom de l'application chargé à l'initialisation de la factory
      //-- peut ne pas être le bon ...
      app = localStorage.getItem('app');
      var langue = localStorage.getItem('current_language');
      if (!langue || langue == null) langue = 'fr';
      return $http.post('/services/' + portalid
        + '/itv/{appname}/initDb?f=json&dsname=' + dsname +
        '&locale=' + langue + '&srid=' + srid, conf
      );
    };


    /**
     * Enregistrement d'un fichier ITV en base de données.
     *
     * @param {*} itvid : IDentifiant de l'ITV de travail
     * @param {object} itvConfig configuration du widget ITV
     * @param {string} itvConfigName nom du fichier de configuration json du widget dans le repo
     * @param {*} automaticcalculation : calcul automatique des scores
     *                                   indigau (rerau)
     * @param {*} agLayers : couches arcgis
     * @param nbQueryParts
     * @param tolerance
     * @param useIndigauEngine
     * @param {*} createConnectionPipes : "true" pour créer les branchements, "false" sinon
     * @param idName
     * @returns {Promise}
     */
    const callSaveItvSce = (itvid, itvConfig, itvConfigName, automaticcalculation,
      agLayers, nbQueryParts, tolerance, useIndigauEngine,
      createConnectionPipes, idName) => {

      // attachVideo est true quand les vidéos doivent être attachés à la portion pour exploitation dans la visualisation
      const attachVideo = gaJsUtils.notNullAndDefined(itvConfig, 'config.importVideo')
          && typeof itvConfig.config.importVideo === 'boolean'
          && itvConfig.config.importVideo;

      let connectionCountField;
      let defaultCountField;
      if (itvConfig.extraconfig) {
        connectionCountField = itvConfig.extraconfig.connectionCountField;
        defaultCountField = itvConfig.extraconfig.defaultCountField;
      }
      if (connectionCountField === undefined) {
        connectionCountField = '';
      }
      if (defaultCountField === undefined) {
        defaultCountField = '';
      }

      // datastore : source de données destinatrice
      // network : réseau d'assainissement à utiliser
      return $http.get(
        '/services/' + portalid + '/itv/{appname}/saveItv?f=json&itvid=' +
        itvid +
        '&datastore=' + itvConfig.config.datastore +
        '&network=' + itvConfig.config.network +
        '&automaticcalculation=' + automaticcalculation +
        '&arcgislayers=' + agLayers +
        (nbQueryParts?'&nbQueryParts='+nbQueryParts:'')+
        '&tolerance=' + tolerance +
        '&indigauEngine=' + !!useIndigauEngine +
        '&connectionCountField=' + connectionCountField +
        '&defaultCountField=' + defaultCountField +
        '&createConnectionPipes=' + createConnectionPipes +
        '&itvConfigName=' + itvConfigName +
        '&idfield='+idName +
        '&attachVideo=' + attachVideo
      );
    };


    /**
     * Soumettre l'ITV.
     *
     * @param {*} itvid Identifiant de l itv
     * @param {*} itvConfig Configuration ITV
     * @param {*} automaticcalculation VRAI/FAUX calculer score rerau
     * @param {*} agLayers Lisste couche arcgis
     * @param {*} createConnectionPipes : "true" pour créer les branchements,
     *                                     "false" sinon
     * @returns promesse du service http
     */
    const saveItv = (itvid, itvConfig, itvConfigName, automaticcalculation,
      agLayers, nbQueryParts, tolerance, useIndigauEngine, createConnectionPipes, idName) => {

      if (!automaticcalculation) {
        return callSaveItvSce(itvid, itvConfig, itvConfigName,
          automaticcalculation, agLayers, nbQueryParts, tolerance, useIndigauEngine,
          createConnectionPipes, idName);
      }
      else {
        let def = $q.defer();
        //-- On veut invoquer le moteur indigau.
        //-- On vérifie d'abord qu'il est fonctionnel.
        IndigauFactory.checkIndigau(app, itvConfig.config.datastore, useIndigauEngine).then(
          (res) => {
            if (res.data.etat == 'erreur') {
              gaDomUtils.hideGlobalLoader();
              swal({
                title: $filter('translate')(
                  'indigau.itv.refreshrerau.' + res.data.errorList[0].message_kis
                ),
                text: res.data.errorMessage,
                type: 'error',
              });
              def.reject();
            }
            else {
              callSaveItvSce(itvid, itvConfig, itvConfigName,
                automaticcalculation, agLayers, nbQueryParts, tolerance, useIndigauEngine,
                createConnectionPipes, idName)
                .then(
                  (res) => {
                    def.resolve(res);
                  },
                  () => {
                    def.reject();
                  }
                );
            }
          }
        );
        return def.promise;
      }
    };


    function uploadzip(f, fname, text, isVideo = false) {
      const promise = $http.post('/services/' +portalid +
          '/itv/{appname}/saveItvzip?f=json&filename=' + f.name +
          '&idfield=' + user.uid + '__' + fname +
          '&multiple=' + false + '&isvideo=' + isVideo,
      text);
      promise.catch((err) => {
        if (err.data && err.data.hasOwnProperty('message') && err.data.hasOwnProperty('detail')
            && typeof err.data.message === 'string' && typeof err.data.detail === 'string') {
          require('toastr').error($filter('translate')(err.data.message) + err.data.detail);
        }
      });
      return promise;
    }


    function uploadzipMultiple(f, fname, a, b, formData, itvId, isVideo = false) {
      const promise = $http.post(
        '/services/{portalid}/itv/uploadfilezip?f=json&filename=' +
          a +
          '_' +
          f.name +
          '&idfield=' +
          itvId +
          '&count=' +
          a +
          '&partcount=' +
          b +
          '&isvideo=' +
          isVideo,
        formData,
        {
          transformRequest: angular.identity,
          headers: { 'Content-Type': undefined },
        }
      );
      promise.catch((err) => {
        if (err.data && err.data.hasOwnProperty('message') && err.data.hasOwnProperty('detail')
            && typeof err.data.message === 'string' && typeof err.data.detail === 'string') {
          require('toastr').error($filter('translate')(err.data.message) + err.data.detail);
        }
      });
      return promise;
    }


    const uploaditv = (f, idfield, networkname, senddata, confName, itvId,
      doMatching, arcGislayers) => {
      var promise = $http.post('/services/' + portalid +
        '/itv/{appname}/startItvUpload?f=json&filename=' + f.name +
        '&idfield=' + idfield +
        '&network=' + networkname +
        '&configname=' + confName +
        '&itvId=' + itvId +
        '&arcgislayers=' + arcGislayers +
        '&doMatching=' + doMatching, senddata
      );
      return promise;
    };


    function getProgression(i) {
      var promise = $http.get(
        '/services/' +
        portalid +
        '/itv/{appname}/getProgression?f=json&threadid=' +
        i
      );
      return promise;
    }

    const getReport = (itvid, tolerance, arcGisLayers, configName) => {
      return $http.get('/services/' + portalid
        + '/itv/{appname}/getReport?f=json&itvid=' + itvid
        + '&configname=' + configName
        + '&arcgislayers=' + arcGisLayers +
        '&tolerance=' + tolerance
      );
    };


    const getPicturesNames = (itvid, network, arcGisLayers, configName) => {
      return $http.get('/services/' + portalid +
        '/itv/{appname}/getPicturesNames?f=json&itvid=' + itvid
        + '&network=' + network
        + '&configname=' + configName
        + '&arcgislayers=' + arcGisLayers
      );
    };

    const getVideosNames = (itvid, network, arcGisLayers, configName) => {
      return $http.get('/services/' + portalid +
        '/itv/{appname}/getVideosNames?f=json&itvid=' + itvid
        + '&network=' + network
        + '&configname=' + configName
        + '&arcgislayers=' + arcGisLayers
      );
    };

    function getPendingItvs() {
      var promise = $http.get(
        '/services/' + portalid + '/itv/{appname}/getPendingItvs?f=json'
      );
      return promise;
    }

    function uploadimage(f, fname, t) {
      gcRestrictionProvider.updateDate();
      gaDomUtils.showGlobalLoader();
      var reader = new FileReader();
      reader.onload = function (e) {
        var text = e.target.result;
        var promise = $http.post(
          '/services/' +
          portalid +
          '/itv/{appname}/saveItvdata?f=json&filename=' +
          f.name +
          '&idfield=' +
          user.uid +
          '__' +
          fname +
          '&dataType=' +
          t,
          text
        );
        promise.then(function (res) {
          // send images
          if (res.data.etat === 'fini') {
            var d = gcRestrictionProvider.lastMessageInSeconde();
            if (d) {
              require('toastr').success(
                $filter('translate')(t === 'videos' ? 'itv.soumission.videoLaunchingSuccess'
                  : 'importwms.upload_succeeded')
              );
              gcRestrictionProvider.updateDate();
            }
          }
          else {
            require('toastr').error(
              $filter('translate')('itv.soumission.errorxmltxt')
            );
          }
        }).finally(
          () => {
            gaDomUtils.hideGlobalLoader();
          }
        );
      };
      reader.readAsDataURL(f);
    }

    /**
     * Convertit une liste de fichiers en tableau de fichiers en base 64 (string)
     * Attend la lecture de tous les fichiers avant de renvoyer le tableau
     * @param event évènement onChange de l'input
     * @return {Promise<string[]>} promesse contenant un tableau de string
     */
    const readAllImages = (event) => {

      // Convertit la liste des fichiers de l'input en tableau et itère dessus
      const files = Array.from(event.target.files).map(file => {

        // Instancie un nouveau fileReader par fichier
        const reader = new FileReader();

        // Créé une nouvelle promesse
        return new Promise(resolve => {

          // Résout la promesse après lecture du fichier
          reader.onload = () => resolve(reader.result);

          // convertit le fichier en string (base64)
          reader.readAsDataURL(file);

        });

      });

      // récupère un tableau contenant tous les fichiers en base64
      return Promise.all(files);
    };


    /**
     * Copie une liste de fichiers dans le dossier de travail de l'ITV du repo (images ou vidéos)
     * Traite l'ensemble des fichiers avant de résoudre la promesse
     * @param files tableau des fichiers chargés dans l'input
     * @param binaryFiles tableau des fichiers convertis en string (base64)
     * @param fname nom du dossier de l'ITV dans le repo
     * @param type type des fichiers ('images' ou 'videos')
     * @return {Promise} contient true si la copie a réussie sinon false
     */
    const uploadImages = (files, binaryFiles, fname, type) => {
      const defer = $q.defer();
      const promises = [];
      if (files && binaryFiles && files instanceof FileList && Array.isArray(binaryFiles)){
        if (files.length !== binaryFiles.length){
          require('toastr').error(
            $filter('translate')('itv.soumission.fileLaunchingError')
          );
          defer.resolve(false);
        }
        else {
          for (let i=0; i < files.length; i++) {
            const promise = $http.post(
              '/services/' +
                portalid +
                '/itv/{appname}/saveItvdata?f=json&filename=' +
                files[i].name +
                '&idfield=' +
                user.uid +
                '__' +
                fname +
                '&dataType=' +
                type,
              binaryFiles[i]
            );
            promises.push(promise);
          }
          $q.all(promises).then(
            res => {
              if (res && Array.isArray(res)){
                const anyError = res.find(
                  result => result.data.etat !== 'fini' || result.status !== 200);
                if (anyError) {
                  require('toastr').error(
                    $filter('translate')('itv.soumission.errorxmltxt')
                  );
                  defer.resolve(false);
                }
                else {
                  const d = gcRestrictionProvider.lastMessageInSeconde();
                  if (d) {
                    require('toastr').success(
                      $filter('translate')
                      (type === 'videos' ? 'itv.soumission.videoLaunchingSuccess'
                        : 'itv.soumission.imageLaunchingSuccess')
                    );
                    gcRestrictionProvider.updateDate();
                    defer.resolve(true);
                  }
                }
              }
            }
          );
        }
      }
      return defer.promise;
    };

    /**
     * Récupère une image en base64
     * @param fname nom du fichier ITV (sans l'extension)
     * @param itvid nom du dossier de travail de l'ITV dans le repo
     * @return {Promise} contenant un objet avec le fichier en base64 dans une propriété 'img'
     */
    function getPicture(fname, itvid) {
      return $http.get(
        '/services/' +
        portalid +
        '/itv/{appname}/getPicture?f=json&filename=' +
        fname +
        '&itvid=' +
        itvid
      );
    }


    /**
     * Construit la liste des couches arcgis sous forme d'une chaine
     * de caractères qui indique la correspondance entre le nom interne
     * de chacune des 4 tables et son pendant arcgis.
     * Exemple: EVT_Inspection_is_Inspections__sep__EVT_Inspection_Part_is_XXX
     *
     * @param {*} conf: configuration qui contient la correspondance
     *                  entre nom interne des tables ITV et nom
     *                  des couches ArcGIS
     */
    const arcgisConfToUrl = (conf) => {
      let urlParam = '';
      const agComponents = conf.arcgiscomponent;

      if (agComponents) {
        for (let ind = 0; ind < lyrNames.length; ind++) {
          if (agComponents[lyrNames[ind]]) {
            if (urlParam !== '') {
              urlParam += '__sep__';
            }
            urlParam += lyrNames[ind] + '_is_' + agComponents[lyrNames[ind]].name;
          }
        }
      }
      return urlParam;
    };


    /**
     * Remplacer l'identification du FTI par le FTI lui même.
     * Le composant de sélection travaille avec le FTI et non son UID.
     *
     * @param {*} conf
     */
    const setAgComponentsAsFti = (conf) => {
      const agComponents = conf.arcgiscomponent;
      if (agComponents) {
        for (let ind = 0; ind < lyrNames.length; ind++) {
          if (agComponents[lyrNames[ind]]) {
            const fti = FeatureTypeFactory.getFeatureByUid(
              agComponents[lyrNames[ind]].ftiUid);
            if (!(agComponents[lyrNames[ind]] instanceof Object)) {
              agComponents[lyrNames[ind]] = {};
            }
            agComponents[lyrNames[ind]].fti = fti;
            agComponents[lyrNames[ind]].name = fti.name;
            agComponents[lyrNames[ind]].alias = fti.alias;
          }
        }
      }
    };


    /**
     * Pour ne pas stocker les FTI complets, on change au moment
     * de l'enregistrement le FTI par son UID et son nom.
     * Toutefois il faut remettre le FTI juste aprés afin de ne pas perturber
     * le fonctionnement du panneau de configuration.
     *
     * @param {*} o
     */
    const simplifyConf = (o) => {
      if (o.arcgiscomponent) {
        let fti;
        for (let ind = 0; ind < lyrNames.length; ind++) {
          if (o.arcgiscomponent[lyrNames[ind]]
            && o.arcgiscomponent[lyrNames[ind]].fti) {
            fti = o.arcgiscomponent[lyrNames[ind]].fti;
            delete o.arcgiscomponent[lyrNames[ind]].fti;
            o.arcgiscomponent[lyrNames[ind]].ftiUid = fti.uid;
          }
        }
      }
    };


    /**
     * Enregistrer la configuration utilisée pour travailler sur les ITVs
     * avec le réseau d'assainissement.
     *
     * @param {*} o : JSON contenant la configuration
     * @param {*} configName : Nom de la configuration
     */
    const saveconf = (o, configName) => {
      var promise = $http.post('/services/' + portalid +
        '/itv/{appname}/saveConf?f=json&configname=' + configName, o
      );
      //-- Enlever les FTI complets de la configuration et les remplacer
      //-- juste par leur UID et leur nom.
      simplifyConf(o);
      promise.then(function (res) {
        setAgComponentsAsFti(o);
        // send images
        if (res.data.etat === 'fini') {
          require('toastr').clear();
          require('toastr').success(
            $filter('translate')('importexportwidget.config.SaveSuccessMsg')
          );
        }
        else {
          require('toastr').clear();
          require('toastr').error(
            $filter('translate')('importexportwidget.config.SaveErrorMsg')
          );
        }
      });
    };


    function getitvPart(id, pn) {
      var promise = $http.get('/services/' + portalid +
        '/itv/{appname}/getItvPart?f=json&itvid=' + id +
        '&partnumber=' + pn
      );
      return promise;
    }


    const getErrors = (id, tolerance, arcgisLayers, configName) =>{
      return $http.get('/services/' + portalid
        + '/itv/{appname}/getErrors?f=json&itvid=' + id
        + '&tolerance=' + tolerance
        + '&configname=' + configName
        + '&arcgislayers=' + arcgisLayers
      );
    };


    /**
     * Enregistrer la correspondance dans le JSON dans le repository
     * du portail.
     * Des appels à cette fonction sont faits fréquemment.
     * Seule est nécessaire l'appel final. Aussi "scope.dontSaveItvPart"
     * est positionné à FAUX au démarrage, puis à VRAI quant l'utime
     * promesse est résolue et le dernier traitement réalisé.
     *
     * @param {*} objValeur
     * @param {*} part
     * @returns
     */
    const saveItvPart = (objValeur, part, dontSaveItvPart, network, datastore) => {
      if (dontSaveItvPart) {
        const def = $q.defer();
        def.resolve();
        return def.promise;
      }
      return itvUtilsFactory.saveItvPart(objValeur.ItvId,
        itvUtilsFactory.getIndTronconFrom(part), part.objValeur,
        network, datastore);
    };




    const ftiObj = {
      code: '',
      value: '',
      featUid: null,
      layerName: '',
      valueMa: null,
      featUidMa: null,
      values: [],
      featUids: [],
      valuesMa: [],
      featUidsMa: [],
      originalValue: '',
      matchedAutomatically: false,
      validated: false,
    };


    const getobj = (obj, code) => {
      const res = angular.copy(obj);
      res.code = code;
      return res;
    };


    const checkITVParts = (p) => {
      let header;
      let headers;
      if (p.objValeur) {
        headers = p.objValeur.headers;
      }
      else {
        headers = p.partHeaders;
      }
      var codes = headers.map(function (header) {
        return header.code;
      });
      if (!codes.includes('CAA') && !codes.includes('@01')) {
        if (!codes.includes('AAA')) {
          header = getobj(ftiObj, 'AAA');
          p.objValeur.headers.push(header);
        }
        if (!codes.includes('AAD') && !codes.includes('AAB')) {
          header = getobj(ftiObj, 'AAD');
          p.objValeur.headers.push(header);
        }
        if (!codes.includes('AAF')) {
          header = getobj(ftiObj, 'AAF');
          p.objValeur.headers.push(header);
        }
      }
      return p;
    };


    /**
     * Recherche du chemin entre 2 regards.
     *
     * @param {*} id1 : identifiant GeoTools du regard de départ
     * @param {*} id2 : identifiant GeoTools du regard d'arrivée
     * @param {*} featUid1 : identifiant du feature Type Info
     *                       du regard de départ
     * @param {*} featUid2 : identifiant du feature Type Info
     *                       du regard d'arrivée
     * @param {*} srid : Spatial Reference IDentifier
     * @param {*} nname : nom du réseau d'assainissement à utiliser pour le
     *                    parcours
     * @returns
     */
    const getPathBetweenNodes = (id1, id2, featUid1, featUid2, srid, nname) => {
      return $http.get('/services/' + portalid +
        '/itv/{appname}/getPathBetweenNodes?f=json&idreg1=' + id1 +
        '&idreg2=' + id2 +'&featuid1=' + featUid1 +'&featuid2=' + featUid2 +
        '&network=' + nname + '&srid=' + srid
      );
    };


    /**
     * Récupére la configuration de gestion des ITVs s'il en existe une.
     * Si un fichier de configuration existe, son contenu est retourné
     * dans la résolution de la promesse. Dans le cas contraire la promesse
     * termine en rejet.
     *
     * @param {*} configName : nom u fichier de configuration
     * @returns promesse
     */
    function getConf(configName) {
      const def = $q.defer();
      const promise = $http.get('/services/' + portalid +
        '/itv/{appname}/getConf?f=json&configname=' + configName
      );
      promise.then((res) => {
        //-- Le retour du service contient un objet qui correspond
        //-- à la classe ItvRet côté JAVA.
        if (res.data.etat === 'erreur') {
          //-- Récupération de la configuration en échec.
          const msg = configName==='creationBranchement'?'itv.noConfigPipe':'itv.noConfig';
          require('toastr').warning($filter('translate')(msg));
          def.reject();
          return;
        }
        //-- La propriété "objValeur" de "ItvRet" contient la configuration.
        const conf = res.data.objValeur;
        setAgComponentsAsFti(conf);
        if (angular.isUndefined(conf.indigau)) {
          conf.indigau = {};
          conf.indigau.withIndigauAutmaticCalc = false;
        }
        def.resolve(conf);
      },
      () => {
        def.reject();
      });
      return def.promise;
    }


    function refreshRerau(filter, indigauEngine, arcGislayers) {
      return $http.get( '/services/' + portalid +
          '/itv/{appname}/refreshindigauscores?f=json&filter=' + filter
        + '&indigauEngine=' + !!indigauEngine + '&arcgislayers=' + arcGislayers
      );
    }


    /**
     * Interrompre le alcul des scores INDIGAU en cours.
     *
     * @param {*} uid : identifiant du thread de calcul
     * @returns
     */
    const interruptRefresh = (uid) => {
      return $http.get('/services/' + portalid +
        '/itv/{appname}/cancelindigauscores?f=json&procid=' + uid);
    };

    /**
     * Récupère l'état du processus de chargement de l'itv unitaire
     * @param threadId identifiant du thread
     * @return {*} processus d'import de l'itv unitaire
     * @see itvSoumission.checkingSoumissionThread
     */
    const getItvLoadingStatus = (threadId) => {
      return $http.get('/services/' + portalid +
        '/itv/{appname}/getItvLoadingStatus?f=json&loadingId=' + threadId);
    };

    /**
     * Récupère la liste des noms des images générées depuis la vidéo
     * Exécutée depuis le bouton "Générer les photos des vidéos"
     * @param itvid identifiant de l'itv avant insertion en bdd
     * @param network nom du réseau ("Assainissement" par exemple)
     * @param datastore nom de la datasource
     * @param arcgisLayers nom des couches ITV dans une source ArcGIS
     * @return {[string]} tableau des noms d'images générées depuis les vidéos
     */
    const generatePicturesFromVideos = (itvid, network, datastore, arcgisLayers, configName) => {
      return $http.get('/services/' + portalid + '/itv/{appname}/generatePicturesFromVideos?f=json'
        + '&itvid=' + itvid
        + '&network=' + network
        + '&datastore=' + datastore
        + '&configname=' + configName
        + '&arcgislayers=' + arcgisLayers
      );
    };

    /**
     * Supprime le dossier de travail de l'itv (repo/ITV/itvName)
     * @param foldername nom du dossier de travail de l'itv
     * @return {boolean} true si la suppression a réussie
     */
    const deletePendingItv = (foldername) => {
      return $http.get('/services/' + portalid +'/itv/{appname}/deletePendingItv?f=json'
          + '&foldername=' + foldername);
    };

    /**
     * Détecte la présence de la variable de configuration "FFMPEG"
     * @return {boolean} true si la variable existe
     */
    const canGeneratePictures = () => {
      return $http.get('/services/' + portalid +'/itv/{appname}/canGeneratePictures?f=json');
    };

    /**
     * Appel le service de vérification de la configuration avec
     * comme paramétres le nom de la source de données,
     * le type de configuration qui indique la partie de configuration
     * à vérifier (par exemple uniquement ce qui concerne le chargement
     * en masse), et la configuration des iTVs compléte.
     *
     * @param {*} conf : configuration de la gestion des ITVs
     * @param {*} confType : partie de la configuration à vérifier
     * @returns promess de l'appel du service
     */
    const checkItvConf = (conf, confType) => {
      return $http.post('/services/' + portalid +
        '/itv/{appname}/checkConf?dsname=' + conf.datastore
        + '&confType=' + confType, conf);
    };


    /**
     * Appel du service de création des branchements décrits dans les ITVs.
     *
     * @param {*} itvIds Objet JSON décrivant les ITVs pour lesquelles
     *                   créer un branchement
     *                   {
     *                    all: true/false,
     *                    ids: [idItv1, idItv2, ..; , idItvN],
     *                    detailsToAvoid: [id1, id1, ... idN],
     *                   }
     * @param {*} datastore Nom de la source de données
     * @param {*} network Nom du réseau assainissement à utiliser
     * @param {*} agLayers Liste des couches arcgis
     * @returns promesse du service HTTP
     */
    const createConnectionPipes = (itvIds, datastore, network, agLayers, configName) => {
      return $http.post('/services/' + portalid +
        '/itv/{appname}/createConnectionPipes?datastore=' + datastore
        + '&arcGisLayers=' + agLayers
        + '&configname=' + configName
        + '&network=' + network, itvIds);
    };



    /**
     * Retourne la valeur du pourcentage à utiliser pour signaler
     * un écart de longueur entre la portion ITV et
     * la ou les canalisations SIG mises en correspondance.
     *
     * @returns tolerance d'ecart de longueur de canalisation
     */
    const getTolerance = (scope) => {
      //-- Valeur de tolérance la plus grande autorisée.
      let tolerance = 0.3;
      if (scope.conf.itvLengthTolerance!==undefined
            && !isNaN(scope.conf.itvLengthTolerance)) {
        //-- Tolérance configurée
        tolerance = scope.conf.itvLengthTolerance / 100;
      }
      return tolerance;
    };




    return {
      itvsmobj: itvsmobj,
      initDb: initDb,
      EVT_full_Inspection: EVT_full_Inspection,
      setEVT_full_Inspection: setEVT_full_Inspection,
      getEVT_full_Inspection: getEVT_full_Inspection,
      saveInspectionProperties: saveInspectionProperties,
      getInspectionProperties: getInspectionProperties,
      saveItv: saveItv,
      getPendingItvs: getPendingItvs,
      getPicturesNames: getPicturesNames,
      getVideosNames: getVideosNames,
      uploadzip: uploadzip,
      uploadzipMultiple: uploadzipMultiple,
      uploaditv: uploaditv,
      uploadimage: uploadimage,
      uploadImages: uploadImages,
      readAllImages: readAllImages,
      saveconf: saveconf,
      getReport: getReport,
      getProgression: getProgression,
      getitvPart: getitvPart,
      checkITVParts: checkITVParts,
      getPicture: getPicture,
      getPathBetweenNodes: getPathBetweenNodes,
      getConf: getConf,
      getErrors: getErrors,
      refreshRerau: refreshRerau,
      interruptRefresh: interruptRefresh,
      arcgisConfToUrl: arcgisConfToUrl,
      getItvLoadingStatus: getItvLoadingStatus,
      generatePicturesFromVideos: generatePicturesFromVideos,
      deletePendingItv: deletePendingItv,
      canGeneratePictures: canGeneratePictures,
      checkItvConf: checkItvConf,
      saveItvPart: saveItvPart,
      getTolerance: getTolerance,
      createConnectionPipes: createConnectionPipes
    };
  };
  itvsoumFactory.$inject = [ '$http', '$rootScope', '$filter',
    'gclayers', 'FeatureTypeFactory', '$q',
    'gcRestrictionProvider', '$timeout', 'IndigauFactory',
    'gaDomUtils', 'itvUtilsFactory', 'gaJsUtils'
  ];
  return itvsoumFactory;
});
