'use strict';
define(function() {
/**
 * Un ensemble de fonctions pour aider à ajuster la hauteur d'une datatable dans
 * un onglet.
 * Pour l'instant, seul est géré le cas où l'onglet est un onglet de la popup Intervention Simple.
 *
 * @typedef {Object} DatatableIsHelper
 * @property {function():number} tabSubpartHeightPadding -
 *    La hauteur totale du padding (haut et bas) de l'éléments
 *    "subpart" à l'intérieur de l'onglet actuel.
 * @property {function(HTMLElement):HTMLElement|null} getRenderInterventionSimle -
 *    Le premier ancêtre de l'élément "element" qui a la classe
 *    "render intervention_simple".
 * @property {function():number} titleHeight -
 *    La hauteur totale des éléments de titre (éléments avec la classe
 *    "title ng-binding") à l'intérieur de l'onglet actuel.
 * @property {function(Object, HTMLElement):number} headerHeight -
 *    La hauteur de l'en-tête de la table.
 * @property {function():number} bottomHeight -
 *    La hauteur totale des éléments "bottomButtons" (les boutons de
 *    bas de page) à l'intérieur de l'onglet actuel.
 */
  const datatableIsHelper = function() {




    /**
     * Trouve le premier ancêtre de l'élément donné qui a la classe donnée.
     *
     * @param {HTMLElement} élément - L'élément à partir duquel commencer la recherche.
     * @param {string} nomDeClasse - Le nom de la classe à rechercher.
     * @returns {HTMLElement|null} L'élément ancêtre avec la classe donnée,
     *                            ou null si aucun élément de ce type n'est trouvé.
     */
    const getParentElement = (element, className) => {
      while (element && element.className && element.className.indexOf(className) === -1) {
        element = element.parentElement;
      }
      return element;
    };


    /**
     * Récupère les éléments qui ont la classe "elementClass" à l'intérieur de
     * "baseElement".
     *
     * @param {HTMLElement} baseElement - L'élément à partir duquel commencer la recherche.
     * @param {string} elementClass - Le nom de la classe à rechercher.
     * @param {HTMLElement[]} res - Un tableau qui contiendra les éléments trouvés.
     *                              Si non fourni, un nouveau tableau sera créé.
     * @return {HTMLElement[]} Les éléments qui ont la classe "elementClass".
     */
    const getElementInsideElement = (baseElement, elementClass, res) => {
      let elementToReturn;
      if (!res) {
        res = [];
      }
      for (let i = 0; i < baseElement.childNodes.length; i++) {
        elementToReturn = baseElement.childNodes[i];
        if (elementToReturn.className
          && elementToReturn.className.indexOf(elementClass) !== -1) {
          res.push(elementToReturn);
        }
        else {
          getElementInsideElement(elementToReturn, elementClass, res);
        }
      }
      return res;
    };


    /**
     * Trouve le premier ancêtre de l'élément "element" qui a la classe
     * "render intervention_simple".
     *
     * @param {HTMLElement} element - L'élément à partir duquel commencer la recherche.
     * @returns {HTMLElement|null} L'élément ancêtre avec la classe "render intervention_simple",
     *                            ou null si aucun élément de ce type n'est trouvé.
     */
    const getRenderInterventionSimle = (element) => {
      return getParentElement(element, 'render intervention_simple');
    };


    /**
     * Renvoie le premier ancêtre de l'élément "datatableElement" qui a la classe
     * "its_onglet". Par exemple, si on est dans la datatable de résultat de recherche,
     * on trouve l'onglet où sont listés les résultats de la recherche.
     *
     * @return {HTMLElement|null} L'élément ancêtre avec la classe "its_onglet",
     *                            ou null si aucun élément de ce type n'est trouvé.
     */
    const getParentTab = (datatableElement) => {
      return getParentElement(datatableElement, 'its_onglet');
    };


    /**
     * Calcule la hauteur que l'on doit soustraire de la hauteur disponible pour
     * afficher la datatable pour laisser de la place pour l'en-tête de la table.
     *
     * L'en-tête de la table a un impact sur la hauteur disponible,
     * si elle est composée d'au moins un des éléments suivants:
     * - des boutons de CRUD (si on a des droits de création ou de suppression)
     * - du bouton d'export CSV
     * - de la mention du nombre total d'enregistrements
     * - du bouton d'import CSV
     *
     * @param {Object} scope - Le scope de la directive gcdatatable.
     * @param {HTMLElement} tableElement - L'élément HTML de la table.
     * @returns {number} La hauteur de l'en-tête de la table, ou 0 si il n'y a pas
     *                  d'en-tête.
     */
    const headerHeight = (scope, tableElement) => {
      // Vérifie si la table a un en-tête (par exemple, pour des actions CRUD ou
      // des options d'export/import).
      const hasHeader = (scope.crud !== undefined && (scope.crud.create || scope.crud.remove))
            || scope.exportCsv === 'true'
            || scope.displayTotalNumber
            || scope.importCsv;

      // Si un en-tête est présent, soustrait sa hauteur de la hauteur disponible.
      if (hasHeader) {
        const tableHeader = tableElement.querySelector('.gcDatatableCrudPourHeight');
        if (tableHeader) {
          // Hauteur de l'en-tête de la table.
          return tableHeader.clientHeight;
        }
      }
      return 0;
    };


/**
 * Calcule la hauteur totale des éléments ayant une classe spécifique
 * à l'intérieur de l'onglet parent du datatable donné.
 *
 * @param {HTMLElement} datatableElement - L'élément datatable à partir duquel
 *                                         commencer la recherche de l'onglet parent.
 * @param {string} elementClass - Le nom de la classe des éléments dont on veut
 *                                calculer la hauteur totale.
 * @returns {number} La hauteur totale des éléments avec la classe spécifiée
 *                   à l'intérieur de l'onglet parent.
 */
    const tabElementHeight = (datatableElement,elementClass) => {
      let height = 0;
      const tabElement = getParentTab(datatableElement);
      if (tabElement) {
        const titleElements = getElementInsideElement(tabElement,elementClass);
        for (const element of titleElements) {
          height += element.clientHeight;
        }
      }
      return height;
    };


    /**
     * Retourne la hauteur totale des éléments de titre (éléments avec la classe
     * "title ng-binding") à l'intérieur de l'onglet actuel.
     * Normalement on ne trouvé qu'un élément titre.
     *
     * @returns {number} La hauteur totale des éléments de titre.
     */
    const titleHeight = (datatableElement) => {
      return tabElementHeight(datatableElement,'title');
    };


    /**
     * Retourne la hauteur totale des éléments "bottomButtons" (les boutons de
     * bas de page) à l'intérieur de l'onglet actuel.
     * Normalement on ne trouvé qu'un élément (la barre de boutons).
     *
     * @returns {number} La hauteur totale des éléments "bottomButtons".
     */
    const bottomHeight = (datatableElement) => {
      return tabElementHeight(datatableElement,'bottomButtons');
    };


    /**
     * Retourne la hauteur totale des éléments de fil d'Ariane (éléments avec la classe
     * "is-breadcrumb") à l'intérieur de l'onglet actuel.
     * Normalement on ne trouvé qu'un élément (le fil d'Ariane).
     *
     * @returns {number} La hauteur totale des éléments de fil d'Ariane.
     */
    const breadCrumbHeight = (datatableElement) => {
      return tabElementHeight(datatableElement,'is-breadcrumb');
    };


    /**
     * Retourne la hauteur totale des éléments de texte (éléments avec la classe
     * "textForDatatableHeight") à l'intérieur de l'onglet actuel.
     * Normalement on ne trouvé qu'un élément (le premier élément de l'onglet).
     *
     * @returns {number} La hauteur totale des éléments de texte.
     */
    const textsHeight = (datatableElement) => {
      return tabElementHeight(datatableElement,'textForDatatableHeight');
    };


    /**
     * Calcule la hauteur totale du padding (haut et bas) de l'éléments
     * "subpart" à l'intérieur de l'onglet actuel. L'élémént "subpart"
     * est le conteneur des éléments de l'onglet.
     * Ceci est utilisé pour ajuster la hauteur du datatable.
     *
     * @returns {number} hauteur total du padding en hauteur
     */
    const tabSubpartHeightPadding = (datatableElement) => {
      const tabElement = getParentTab(datatableElement);
      if (tabElement) {
        const subpartElements
          = getElementInsideElement(tabElement,'subpart');
        for (const element of subpartElements) {
          const style = getComputedStyle(element);
          return parseInt(style.paddingTop) + parseInt(style.paddingBottom);
        }
      }
      return 0;
    };




    return {
      tabSubpartHeightPadding, getRenderInterventionSimle,
      titleHeight, headerHeight, bottomHeight, getParentTab, breadCrumbHeight,
      textsHeight
    };
  };
  datatableIsHelper.$inject = [];
  return datatableIsHelper;
});
